/* -*- mode: c++; c-basic-offset: 4 -*- */

#include <cstdint>
#include <string>
#include <iostream>
#include <cstdlib>
#include <string_view>
#include <set>
#include <list>
#include <array>
#include <map>
#include <sstream>
#include <iomanip>
#include <fstream>
#include <cstring>
#include <algorithm>
#include <cctype>
#include <random>
#include <vector>

constexpr int RUN_OK               = 0;
constexpr int RUN_PRESENTATION_ERR = 4;
constexpr int RUN_WRONG_ANSWER_ERR = 5;
constexpr int RUN_CHECK_FAILED     = 6;

/*
     12 13
     14 15
     06 07
  08 00 01 10
  09 02 03 11
     04 05
 */

// operation permutations
// L                    0, 1, 2, 3, 4, 5, 6, 7, 8, 9,10,11,12,13,14,15
const uint8_t oper_L[]{12, 1,14, 3, 6, 5, 4, 7, 9, 8,10,11, 0,13, 2,15};
// R                    0, 1, 2, 3, 4, 5, 6, 7, 8, 9,10,11,12,13,14,15
const uint8_t oper_R[]{ 0,13, 2,15, 4, 7, 6, 5, 8, 9,11,10,12, 1,14, 3};
// U                    0, 1, 2, 3, 4, 5, 6, 7, 8, 9,10,11,12,13,14,15
const uint8_t oper_U[]{15,14, 2, 3, 4, 5, 7, 6,10, 9, 8,11,12,13, 1, 0};
// D                    0, 1, 2, 3, 4, 5, 6, 7, 8, 9,10,11,12,13,14,15
const uint8_t oper_D[]{ 0, 1,13,12, 5, 4, 6, 7, 8,11,10, 9, 3, 2,14,15};
// F                    0, 1, 2, 3, 4, 5, 6, 7, 8, 9,10,11,12,13,14,15
const uint8_t oper_F[]{ 1, 3, 0, 2, 8, 9,10,11, 7, 6, 5, 4,14,12,15,13};

const uint8_t * opers[256]{};

constexpr std::string_view START("1111223344556666");

bool is_valid(const std::string &str)
{
    int count[6]{};
    if (str.length() != 16) {
        return false;
    }
    for (auto c : str) {
        if (c < '1' || c > '6')
            return false;
        ++count[c - '1'];
    }
    return count[0] == 4 && count[1] == 2 && count[2] == 2 && count[3] == 2 && count[4] == 2 && count[5] == 4;
}

std::string do_move(const std::string &state, char m)
{
    const uint8_t *oper = opers[(uint8_t) m];
    if (!oper) abort();
    std::string res(16, ' ');
    for (int i = 0; i < 16; ++i) {
        res[oper[i]] = state[i];
    }
    return res;
}

struct Transition
{
    std::string new_state;
    std::string prev_state;
    char mov{' '};

    Transition(std::string n, std::string p, char m)
        : new_state(std::move(n)),
          prev_state(std::move(p)),
          mov(m)
    {
    }
    Transition(const Transition &o)
        :new_state(o.new_state),
         prev_state(o.prev_state),
         mov(o.mov)
    {
    }
    Transition(Transition &&o) noexcept
        :new_state(std::move(o.new_state)),
         prev_state(std::move(o.prev_state)),
         mov(o.mov)
    {
    }
};

std::map<std::string, int>
generate_states()
{
    std::list<std::pair<std::string,int>> queue;
    queue.emplace_back(START, 0);
    std::map<std::string, int> reachable;
    while (queue.size() > 0) {
        auto [ cur, dist] = queue.front();
        queue.pop_front();
        if (reachable.count(cur) > 0) continue;
        reachable.emplace(cur, dist);
        static const std::array<char, 5> valids{'L','R','U','D','F'};
        for (auto m : valids) {
            std::string next = do_move(cur, m);
            if (!reachable.count(next)) {
                queue.emplace_back(next, dist + 1);
            }
        }
    }
    return reachable;
}

std::string
find_movements(const std::string &begstate, const std::string &endstate)
{
    std::list<Transition> queue;
    queue.emplace_back(begstate, "", ' ');
    std::map<std::string, Transition> reachable;
    while (queue.size() > 0) {
        auto t = std::move(queue.front());
        queue.pop_front();
        if (reachable.count(t.new_state) > 0) continue;
        reachable.emplace(t.new_state, t);
        if (t.new_state == endstate)
            break;
        static const std::array<char, 5> valids{'L','R','U','D','F'};
        for (auto m : valids) {
            std::string next = do_move(t.new_state, m);
            if (!reachable.count(next)) {
                queue.emplace_back(next, t.new_state, m);
            }
        }
    }
    auto ist = reachable.find(std::string(endstate));
    std::string recover;
    if (ist != reachable.end()) {
        while (ist->second.mov != ' ') {
            recover.push_back(ist->second.mov);
            ist = reachable.find(ist->second.prev_state);
            if (ist == reachable.end()) abort();
        }
    } else {
        abort();
    }
    std::reverse(recover.begin(), recover.end());
    return recover;
}

int main(int argc, char *argv[])
{
    char *pn = strrchr(argv[0], '/');
    if (pn) ++pn;
    else pn = argv[0];
    std::string_view name(pn);

    opers['L'] = oper_L;
    opers['R'] = oper_R;
    opers['U'] = oper_U;
    opers['D'] = oper_D;
    opers['F'] = oper_F;

    if (name == "generate_states" || name == "generate_tests") {
        auto reachable = generate_states();
        if (name == "generate_states") {
            for (const auto &s : reachable) {
                std::cout << s.first << " " << s.second << std::endl;
            }
        } else if (name == "generate_tests") {
            int serial = 2;
            for (const auto &s : reachable) {
                std::ostringstream ds;
                ds << "tests/" << std::setw(3) << std::setfill('0') << serial << ".dat";
                std::ofstream dss(ds.str());
                dss << s.first << std::endl;
                ++serial;
            }
        }
        return 0;
    }
    if (name == "generate9") {
        if (argc != 2) {
            std::cerr << "seed required" << std::endl;
            return 1;
        }
        auto reachable = generate_states();
        std::mt19937 rnd(std::stoul(argv[1]));
        std::vector<std::string> sts;
        for (const auto &p : reachable) {
            sts.push_back(p.first);
        }
        for (int serial = 2; serial < 52; ++serial) {
            int ind1, ind2;
            do {
                ind1 = rnd() % sts.size();
                ind2 = rnd() % sts.size();
            } while (ind1 == ind2);
            auto begstate = sts[ind1];
            auto endstate = sts[ind2];
            auto tr = find_movements(begstate, endstate);

            std::ostringstream ds;
            ds << "tests9/" << std::setw(3) << std::setfill('0') << serial << ".dat";
            std::ofstream dss(ds.str());
            dss << begstate << std::endl;
            dss << tr << std::endl;

            std::ostringstream as;
            as << "tests9/" << std::setw(3) << std::setfill('0') << serial << ".ans";
            std::ofstream ass(as.str());
            ass << endstate << std::endl;
        }
        return 0;
    }

    if (name == "check") {
        if (argc != 4) {
            std::cerr << "wrong number of arguments" << std::endl;
            return RUN_CHECK_FAILED;
        }
        std::ifstream it(argv[1]);
        std::string begin;
        it >> begin;
        if (!is_valid(begin)) {
            std::cerr << "initial state '" << begin << "' is invalid" << std::endl;
            return RUN_CHECK_FAILED;
        }
        std::ifstream as(argv[3]);
        std::string good_moves;
        as >> good_moves;

        std::ifstream os(argv[2]);
        std::string check_moves;
        char m;
        while (os.get(m)) {
            if (isspace((unsigned char) m)) {
                // just skip
            } else if (m == 'L' || m == 'R' || m == 'U' || m == 'D' || m == 'F') {
                check_moves.push_back(m);
            } else {
                std::cerr << "invalid character in output" << std::endl;
                return RUN_PRESENTATION_ERR;
            }
        }
        if (check_moves.length() > 1000) {
            std::cerr << "too many moves" << std::endl;
            return RUN_WRONG_ANSWER_ERR;
        }
        for (auto m : check_moves) {
            begin = do_move(begin, m);
        }
        if (begin != START) {
            std::cerr << "moves end in wrong state " << begin << std::endl;
            return RUN_WRONG_ANSWER_ERR;
        }
        if (check_moves.length() < good_moves.length()) {
            std::cerr << "participant found a better answer" << std::endl;
            return RUN_CHECK_FAILED;
        }
        if (check_moves.length() > good_moves.length()) {
            std::cerr << "participant found a bad answer" << std::endl;
            return RUN_WRONG_ANSWER_ERR;
        }
        std::cerr << "ok" << std::endl;
        return 0;
    }

    std::string start;
    std::cin >> start;
    std::cout << find_movements(start, std::string(START)) << std::endl;
}
